<?php
session_start();
date_default_timezone_set('America/Sao_Paulo');

require '../../class/configClass.php';
Config::load('../../config/config.php');
include_once('../../conexao/conexao.php');
include_once('../../funcoes/general.php');

require '../../class/walletClass.php';
require '../../class/getPlanosConfig.php';
require '../../class/vendasClass.php';
require '../../class/produtosClass.php';

// Verifica se está logado ou não
if (!isset($_SESSION['usuario']['email'])) {
    header('Location: /login');
    exit();
}

$customBase = __DIR__ . '/../../custom/includes/';
if (isset($_SESSION['usuario']['is_adm']) && $_SESSION['usuario']['is_adm'] == 1) {
    $customDir = $customBase . 'admin/';
} else {
    $customDir = $customBase . 'user/';
}

if (is_dir($customDir)) {
    foreach (glob($customDir . '*.php') as $filename) {
        include_once $filename;
    }
}

if ($_SESSION['usuario']['is_adm'] != '1') {
    header('Location: /dash/dashboard');
    exit();
}

// Obter o nome do whitelabel atual
$whitelabelAtual = strtolower(Config::get('site_configs.name'));
$fornecedorAtual = pegaPrefixo() . $_SESSION['usuario']['id_usuario'];
$fornecedorAtual = strtolower($fornecedorAtual);

// Definir filtros padrão
$filtroDataDe = isset($_GET['data_de']) && !empty($_GET['data_de']) ? $_GET['data_de'] : '';
$filtroDataAte = isset($_GET['data_ate']) && !empty($_GET['data_ate']) ? $_GET['data_ate'] : '';
$filtroIntegracao = isset($_GET['integracao']) ? $_GET['integracao'] : '';
$filtroStatusPago = isset($_GET['status_pago']) ? $_GET['status_pago'] : '';
$filtroIdVenda = isset($_GET['id_venda']) && !empty($_GET['id_venda']) ? $_GET['id_venda'] : '';

// Query para buscar opções de integração disponíveis para este fornecedor
$sqlIntegracoes = "SELECT DISTINCT integracao FROM drophub_global.minhas_vendas WHERE LOWER(fornecedor) = :fornecedor ORDER BY integracao";
$stmtIntegracoes = $pdo->prepare($sqlIntegracoes);
$stmtIntegracoes->bindParam(':fornecedor', $fornecedorAtual, PDO::PARAM_STR);
$stmtIntegracoes->execute();
$integracoes = $stmtIntegracoes->fetchAll(PDO::FETCH_COLUMN);

// Construir a query com filtros - buscar todas as vendas do fornecedor
$whereConditions = ["LOWER(fornecedor) = :fornecedor"];
$params = ['fornecedor' => $fornecedorAtual];

// Adicionar filtros se fornecidos
if (!empty($filtroDataDe)) {
    $whereConditions[] = "vendido_dia >= :data_de";
    $params['data_de'] = $filtroDataDe . ' 00:00:00';
}

if (!empty($filtroDataAte)) {
    $whereConditions[] = "vendido_dia <= :data_ate";
    $params['data_ate'] = $filtroDataAte . ' 23:59:59';
}

if (!empty($filtroIntegracao)) {
    $whereConditions[] = "integracao = :integracao";
    $params['integracao'] = $filtroIntegracao;
}

if (!empty($filtroStatusPago)) {
    if ($filtroStatusPago === 'pago') {
        $whereConditions[] = "vlr_custo_pago = 1";
    } else if ($filtroStatusPago === 'nao_pago') {
        $whereConditions[] = "vlr_custo_pago = 0";
    }
}

$filtroSemEtiqueta = isset($_GET['sem_etiqueta']) ? $_GET['sem_etiqueta'] : '';
if (!empty($filtroSemEtiqueta)) {
    $whereConditions[] = "etiqueta_ml IN ('processing_label', 'buffering', 'buffered')";
    $whereConditions[] = "(reembolsado = 0 OR reembolsado IS NULL)";
    $whereConditions[] = "(cancelado = 0 OR cancelado IS NULL)";
    $whereConditions[] = "(observacoes NOT LIKE '%SHOPEE: Pedido já enviado%' OR observacoes IS NULL)";
    $whereConditions[] = "(observacoes NOT LIKE '%ERRO ETIQUETA MercadoLivre:%' OR observacoes IS NULL)";
    $whereConditions[] = "(LOWER(observacoes) NOT LIKE '%erro 338a%' OR observacoes IS NULL)";
    $whereConditions[] = "(LOWER(observacoes) NOT LIKE '%erro 388a%' OR observacoes IS NULL)";
    $whereConditions[] = "(observacoes NOT LIKE '%HTTP 200: Este pedido já foi enviado ao cliente%' OR observacoes IS NULL)";
    $whereConditions[] = "(shipping_id IS NOT NULL AND shipping_id != 0)";
    $whereConditions[] = "(observacoes not like '%Esta venda está para combinar o envio com o fornecedor%')";
    $whereConditions[] = "(observacoes not like '%HTTP 200: Erro 338A%')";
}

if (!empty($filtroIdVenda)) {
    $whereConditions[] = "saleTitle LIKE :id_venda";
    $params['id_venda'] = '%' . $filtroIdVenda . '%';
}

$whereClause = implode(' AND ', $whereConditions);

// Paginação
$itensPorPagina = 12;
$paginaAtual = isset($_GET['pagina']) ? max(1, (int)$_GET['pagina']) : 1;
$offset = ($paginaAtual - 1) * $itensPorPagina;

// Query para contar total de registros
$sqlContagem = "SELECT COUNT(*) FROM drophub_global.minhas_vendas WHERE $whereClause";
$stmtContagem = $pdo->prepare($sqlContagem);
$stmtContagem->execute($params);
$totalRegistros = $stmtContagem->fetchColumn();
$totalPaginas = ceil($totalRegistros / $itensPorPagina);

// Contar vendas pagas e não pagas (sem filtro de status para ter o total real)
$whereConditionsResumo = ["LOWER(fornecedor) = :fornecedor"];
$paramsResumo = ['fornecedor' => $fornecedorAtual];

// Adicionar filtros de data e integração se existirem (mas não o filtro de status)
if (!empty($filtroDataDe)) {
    $whereConditionsResumo[] = "vendido_dia >= :data_de";
    $paramsResumo['data_de'] = $filtroDataDe . ' 00:00:00';
}
if (!empty($filtroDataAte)) {
    $whereConditionsResumo[] = "vendido_dia <= :data_ate";
    $paramsResumo['data_ate'] = $filtroDataAte . ' 23:59:59';
}
if (!empty($filtroIntegracao)) {
    $whereConditionsResumo[] = "integracao = :integracao";
    $paramsResumo['integracao'] = $filtroIntegracao;
}
if (!empty($filtroIdVenda)) {
    $whereConditionsResumo[] = "saleTitle LIKE :id_venda";
    $paramsResumo['id_venda'] = '%' . $filtroIdVenda . '%';
}

$whereClauseResumo = implode(' AND ', $whereConditionsResumo);

// Contar vendas pagas
$sqlVendasPagas = "SELECT COUNT(*) FROM drophub_global.minhas_vendas WHERE $whereClauseResumo AND vlr_custo_pago = 1";
$stmtVendasPagas = $pdo->prepare($sqlVendasPagas);
$stmtVendasPagas->execute($paramsResumo);
$totalVendasPagas = $stmtVendasPagas->fetchColumn();

// Contar vendas não pagas
$sqlVendasNaoPagas = "SELECT COUNT(*) FROM drophub_global.minhas_vendas WHERE $whereClauseResumo AND vlr_custo_pago = 0";
$stmtVendasNaoPagas = $pdo->prepare($sqlVendasNaoPagas);
$stmtVendasNaoPagas->execute($paramsResumo);
$totalVendasNaoPagas = $stmtVendasNaoPagas->fetchColumn();

// Contar vendas sem etiqueta
$sqlVendasSemEtiqueta = "SELECT COUNT(*) FROM drophub_global.minhas_vendas WHERE $whereClauseResumo AND etiqueta_ml IN ('processing_label', 'buffering', 'buffered') AND (reembolsado = 0 OR reembolsado IS NULL) AND (cancelado = 0 OR cancelado IS NULL) AND (observacoes NOT LIKE '%SHOPEE: Pedido já enviado%' OR observacoes IS NULL) AND (LOWER(observacoes) NOT LIKE '%erro 338a%' OR observacoes IS NULL) AND (LOWER(observacoes) NOT LIKE '%erro 388a%' OR observacoes IS NULL) AND (observacoes NOT LIKE '%HTTP 200: Este pedido já foi enviado ao cliente%' OR observacoes IS NULL) AND (observacoes NOT LIKE '%ERRO ETIQUETA MercadoLivre:%' OR observacoes IS NULL) AND (shipping_id IS NOT NULL AND shipping_id != 0) AND (observacoes not like '%Esta venda está para combinar o envio com o fornecedor%') AND (observacoes not like '%HTTP 200: Erro 338A%')";
$stmtVendasSemEtiqueta = $pdo->prepare($sqlVendasSemEtiqueta);
$stmtVendasSemEtiqueta->execute($paramsResumo);
$totalVendasSemEtiqueta = $stmtVendasSemEtiqueta->fetchColumn();

// Total geral (para validação)
$totalGeralVendas = $totalVendasPagas + $totalVendasNaoPagas;

// Query para buscar todas as vendas do fornecedor
$sqlVendas = "SELECT * FROM drophub_global.minhas_vendas WHERE $whereClause ORDER BY vendido_dia DESC LIMIT :limit OFFSET :offset";
$stmtVendas = $pdo->prepare($sqlVendas);
foreach ($params as $key => $value) {
    $stmtVendas->bindValue(":$key", $value);
}
$stmtVendas->bindValue(':limit', $itensPorPagina, PDO::PARAM_INT);
$stmtVendas->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmtVendas->execute();
$vendas = $stmtVendas->fetchAll(PDO::FETCH_ASSOC);

// Se o filtro sem_etiqueta estiver ativo, buscar todos os IDs para o botão de reprocessar
$idsSemEtiqueta = [];
if (!empty($filtroSemEtiqueta)) {
    $sqlIds = "SELECT saleTitle FROM drophub_global.minhas_vendas WHERE $whereClause";
    $stmtIds = $pdo->prepare($sqlIds);
    foreach ($params as $key => $value) {
        $stmtIds->bindValue(":$key", $value);
    }
    $stmtIds->execute();
    $idsSemEtiqueta = $stmtIds->fetchAll(PDO::FETCH_COLUMN);
}


// Valida Sessoes e atualiza dados
validarSessaoUsuario($pdo, $_SESSION['usuario']['id_usuario']);

?>


<!DOCTYPE html>
<html lang="en" class="light-style layout-navbar-fixed layout-menu-fixed layout-compact" dir="ltr"
    data-theme="theme-default" data-assets-path="../../assets/"
    data-template="vertical-menu-template-no-customizer-starter" >
<head>
<?php include('../../includes/dashHead.php'); ?>

    <link rel="stylesheet" href="../../assets/vendor/libs/apex-charts/apex-charts.css" />
    <link rel="stylesheet" href="../../assets/vendor/libs/swiper/swiper.css" />
    <link rel="stylesheet" href="../../assets/vendor/css/pages/cards-advance.css" />
    <link rel="stylesheet" href="../../assets/vendor/libs/flatpickr/flatpickr.css" />

    <!-- Helpers -->
    <script src="../../assets/vendor/js/helpers.js"></script>
    <script src="../../assets/vendor/js/template-customizer.js"></script>
    <script src="../../assets/js/config.js"></script>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/notyf@3/notyf.min.css">
    <script src="https://cdn.jsdelivr.net/npm/notyf@3/notyf.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="../../assets/js/funcoes.js"></script>

</head>

<body>

    <!-- Layout wrapper -->
    <div class="layout-wrapper layout-content-navbar">
        <div class="layout-container">

            <aside id="layout-menu" class="layout-menu menu-vertical menu bg-menu-theme">
                <div class="app-brand demo mt-3">
                    <a href="dashboard" class="app-brand-link">

                        <?php
                        if (Config::get('site_configs.text_icon') == true) {
                        ?>
                        <span class="app-brand-logo">
                            <img src="<?php echo Config::get('site_configs.icon'); ?>"
                                width="<?php echo Config::get('site_configs.icon_width'); ?>"
                                height="<?php echo Config::get('site_configs.icon_height'); ?>"
                                style="object-fit: contain;">
                        </span>
                        <span
                            class="app-brand-text text-body fw-bold ms-1"><?php echo Config::get('site_configs.name'); ?></span>
                        <?php
                        } else {
                        ?>
                        <span class="app-brand-logo">
                            <img src="<?php echo Config::get('site_configs.icon'); ?>"
                                width="<?php echo Config::get('site_configs.icon_width'); ?>"
                                height="<?php echo Config::get('site_configs.icon_height'); ?>"
                                style="object-fit: contain;">
                        </span>
                        <?php
                        }
                        ?>

                    </a>

                    <a href="javascript:void(0);" class="layout-menu-toggle menu-link text-large ms-auto">
                        <i class="ti menu-toggle-icon d-none d-xl-block ti-sm align-middle"></i>
                        <i class="ti ti-x d-block d-xl-none ti-sm align-middle"></i>
                    </a>
                </div>

                <div class="menu-inner-shadow"></div>

                <ul class="menu-inner py-1">
                    <?php
                    if ($_SESSION['usuario']['n_acesso'] == '0') {
                        include('../../includes/sidebars-menus/sidebar_usuario.php');
                    } else {
                        include('../../includes/sidebars-menus/sidebar_master.php');
                    }
                    ?>

                </ul>
            </aside>
            <!-- / Menu -->

            <!-- Layout container -->
            <div class="layout-page">
                <!-- Navbar -->
                <nav class="layout-navbar container-xxl navbar navbar-expand-xl navbar-detached align-items-center bg-navbar-theme"
                    id="layout-navbar">
                    <div class="layout-menu-toggle navbar-nav align-items-xl-center me-3 me-xl-0 d-xl-none">
                        <a class="nav-item nav-link px-0 me-xl-4" href="javascript:void(0)">
                            <i class="ti ti-menu-2 ti-sm"></i>
                        </a>
                    </div>

                    <div class="navbar-nav-right d-flex align-items-center" id="navbar-collapse">
                        <?php include('../../includes/dashnav.php') ?>

                    </div>
                </nav>

                <!-- / Navbar -->

                <!-- Content wrapper -->
                <div class="content-wrapper">
                    <div class="container-xxl flex-grow-1 container-p-y">
                        <!--
                            <?php
                            echo '<pre>';
                            $dump_sessions = json_encode($_SESSION, JSON_PRETTY_PRINT);
                            echo $dump_sessions;
                            echo '</pre>';
                            ?>
                        -->

                        <div class="container-xxl flex-grow-1 container-p-y">
                            <div class="mt-3">
                                <div class="card rounded shadow">
                                    <div class="card-header d-flex justify-content-between align-items-center">
                                        <h5 class="card-title mb-1">Todas as Vendas</h5>
                                        <span class="badge bg-label-primary"><?= $totalRegistros ?> vendas
                                            encontradas</span>
                                    </div>

                                    <div class="card-body mt-3">
                                        <!-- Formulário de Filtros -->
                                        <form method="GET" class="mb-4">
                                            <div class="row g-3">
                                                <div class="col-md-3">
                                                    <label for="id_venda" class="form-label">ID da Venda:</label>
                                                    <input type="text" name="id_venda" id="id_venda"
                                                        value="<?= htmlspecialchars($filtroIdVenda) ?>"
                                                        class="form-control" placeholder="Digite o ID da venda">
                                                </div>
                                                <div class="col-md-3">
                                                    <label for="periodo" class="form-label">Período:</label>
                                                    <input type="text" id="periodo" class="form-control"
                                                        placeholder="Selecione o período" autocomplete="off" />
                                                    <input type="hidden" name="data_de" id="data_de"
                                                        value="<?= htmlspecialchars($filtroDataDe) ?>">
                                                    <input type="hidden" name="data_ate" id="data_ate"
                                                        value="<?= htmlspecialchars($filtroDataAte) ?>">
                                                </div>
                                                <div class="col-md-3">
                                                    <label for="integracao" class="form-label">Integração:</label>
                                                    <select name="integracao" id="integracao" class="form-control">
                                                        <option value="">Todas as integrações</option>
                                                        <?php foreach ($integracoes as $integracao): ?>
                                                        <option value="<?= htmlspecialchars($integracao) ?>"
                                                            <?= $filtroIntegracao === $integracao ? 'selected' : '' ?>>
                                                            <?= htmlspecialchars($integracao) ?>
                                                        </option>
                                                        <?php endforeach; ?>
                                                    </select>
                                                </div>
                                                <div class="col-md-2">
                                                    <label for="status_pago" class="form-label">Status
                                                        Pagamento:</label>
                                                    <select name="status_pago" id="status_pago" class="form-control">
                                                        <option value="">Todos os status</option>
                                                        <option value="pago"
                                                            <?= $filtroStatusPago === 'pago' ? 'selected' : '' ?>>Pago
                                                        </option>
                                                        <option value="nao_pago"
                                                            <?= $filtroStatusPago === 'nao_pago' ? 'selected' : '' ?>>
                                                            Não Pago</option>
                                                    </select>
                                                </div>
                                            </div>
                                            <div class="row mt-3">
                                                <div class="col-12">
                                                    <button type="submit" class="btn btn-primary">
                                                        <i class="ti ti-search"></i> Filtrar
                                                    </button>
                                                    <a href="?" class="btn btn-secondary">
                                                        <i class="ti ti-refresh"></i> Limpar Filtros
                                                    </a>
                                                    <button type="button" class="btn btn-success" id="btnGerarRelatorio">
                                                        <i class="ti ti-file-spreadsheet"></i> Gerar Relatório
                                                    </button>
                                                    <?php if (!empty($filtroSemEtiqueta)): ?>
                                                    <button type="button" class="btn btn-warning" id="btnReprocessarEtiquetas">
                                                        <i class="ti ti-refresh"></i> Reprocessar Todas Etiquetas
                                                    </button>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </form>

                                        <hr>

                                        <!-- Lista de Vendas -->
                                        <?php if (!empty($vendas)): ?>
                                        <div class="row">
                                            <?php foreach ($vendas as $venda): ?>
                                            <div class="col-md-6 col-lg-4 mb-3">
                                                <div class="card h-100">
                                                    <div class="card-body p-3">
                                                        <div class="d-flex justify-content-between align-items-start mb-2">
                                                            <h6 class="card-title mb-0">
                                                                <?= htmlspecialchars(substr($venda['saleTitle'], 0, 30)) ?><?= strlen($venda['saleTitle']) > 30 ? '...' : '' ?>
                                                            </h6>
                                                            <div class="d-flex flex-wrap gap-1 justify-content-end" style="max-width: 60%;">
                                                                <?php
                                                                // Prioridade: Reembolsado > Enviado > Status de Pagamento
                                                                if (isset($venda['reembolsado']) && $venda['reembolsado'] == 1): ?>
                                                                    <span class="badge bg-danger" style="font-size: 0.7rem;">
                                                                        <i class="ti ti-arrow-back-up me-1"></i>Reembolsado
                                                                    </span>
                                                                <?php elseif (isset($venda['cancelado']) && $venda['cancelado'] == 1): ?>
                                                                    <span class="badge bg-danger" style="font-size: 0.7rem;">
                                                                        <i class="ti ti-x me-1"></i>Cancelado
                                                                    </span>
                                                                <?php elseif (isset($venda['enviado']) && $venda['enviado'] == 1): ?>
                                                                    <span class="badge bg-info" style="font-size: 0.7rem;">
                                                                        <i class="ti ti-truck me-1"></i>Enviado
                                                                    </span>
                                                                <?php elseif (strpos($venda['observacoes'] ?? '', 'SHOPEE: Pedido já enviado') !== false): ?>
                                                                    <span class="badge bg-info" style="font-size: 0.7rem;">
                                                                        <i class="ti ti-truck me-1"></i>Pedido já enviado
                                                                    </span>
                                                                <?php elseif (strpos($venda['observacoes'] ?? '', 'Erro 388A') !== false || strpos($venda['observacoes'] ?? '', 'HTTP 200: Este pedido já foi enviado ao cliente') !== false): ?>
                                                                    <span class="badge bg-danger" style="font-size: 0.7rem;">
                                                                        <i class="ti ti-alert-triangle me-1"></i>Pedido já enviado - Ver Observação
                                                                    </span>
                                                                <?php elseif (strpos($venda['observacoes'] ?? '', 'ERRO ETIQUETA MercadoLivre') !== false): ?>
                                                                    <span class="badge bg-danger" style="font-size: 0.7rem;">
                                                                        <i class="ti ti-alert-triangle me-1"></i>Ver Observação
                                                                    </span>
                                                                <?php else: ?>
                                                                    <?php if ($venda['vlr_custo_pago'] == 1): ?>
                                                                        <span class="badge bg-success" style="font-size: 0.7rem;">
                                                                            <i class="ti ti-circle-check me-1"></i>Pago
                                                                        </span>
                                                                    <?php else: ?>
                                                                        <span class="badge bg-warning" style="font-size: 0.7rem;">
                                                                            <i class="ti ti-currency-dollar me-1"></i>Pendente
                                                                        </span>
                                                                    <?php endif; ?>

                                                                    <?php if ($venda['etiqueta_ml'] == 'processing_label'): ?>
                                                                        <span class="badge bg-warning" style="font-size: 0.7rem;">
                                                                            <i class="ti ti-alert-circle me-1"></i>Sem Etiqueta
                                                                        </span>
                                                                    <?php elseif (($venda['etiqueta_ml'] == 'buffering' || $venda['etiqueta_ml'] == 'buffered') && strpos($venda['observacoes'] ?? '', 'ERRO ETIQUETA MercadoLivre') === false): ?>
                                                                        <span class="badge bg-warning" style="font-size: 0.7rem;">
                                                                            <i class="ti ti-clock me-1"></i>Aguardando Liberação
                                                                        </span>
                                                                    <?php endif; ?>
                                                                <?php endif; ?>
                                                            </div>
                                                        </div>

                                                        <div class="mb-2 d-flex flex-column">
                                                            <small class="text-muted">
                                                                <i class="ti ti-calendar"></i>
                                                                Vendido:
                                                                <?= date('d/m/Y H:i', strtotime($venda['vendido_dia'])) ?>
                                                            </small>

                                                            <?php if ($venda['vlr_custo_pago'] == 1 && !empty($venda['dta_vl_pago'])): ?>
                                                            <small class="text-success">
                                                                <i class="ti ti-cash"></i>
                                                                Pago:
                                                                <?= date('d/m/Y H:i', strtotime($venda['dta_vl_pago'])) ?>
                                                            </small>
                                                            <?php endif; ?>

                                                            <?php if ($venda['enviado'] == 1 && !empty($venda['dta_enviado'])): ?>
                                                            <small class="text-info">
                                                                <i class="ti ti-truck"></i>
                                                                Enviado:
                                                                <?= date('d/m/Y H:i', strtotime($venda['dta_enviado'])) ?>
                                                            </small>
                                                            <?php endif; ?>
                                                        </div>

                                                        <div class="mb-2">
                                                            <span
                                                                class="badge bg-label-primary"><?= htmlspecialchars($venda['integracao']) ?></span>
                                                            <?php if (!empty($venda['tipo_envio'])): ?>
                                                            <span
                                                                class="badge bg-label-info"><?= htmlspecialchars($venda['tipo_envio']) ?></span>
                                                            <?php endif; ?>
                                                        </div>

                                                        <!-- Miniaturas dos produtos -->
                                                        <div class="miniaturas-produtos mb-2"
                                                            data-sale-title="<?= htmlspecialchars($venda['saleTitle']) ?>">
                                                            <div class="miniaturas-loading text-muted small">Carregando
                                                                produtos...</div>
                                                        </div>

                                                        <div class="mb-2">
                                                            <?php if (!empty($venda['vlr_custo'])): ?>
                                                            <br>
                                                            <small class="text-muted">
                                                                <i class="ti ti-coin"></i>
                                                                Custo: R$
                                                                <?= number_format($venda['vlr_custo'], 2, ',', '.') ?>
                                                            </small>
                                                            <?php endif; ?>
                                                        </div>

                                                        <div class="mt-2">
                                                            <button type="button"
                                                                class="btn btn-outline-primary btn-sm w-100 ver-detalhes"
                                                                data-sale-title="<?= htmlspecialchars($venda['saleTitle']) ?>"
                                                                data-sale-id="<?= htmlspecialchars($venda['saleId']) ?>"
                                                                data-order-id="<?= htmlspecialchars($venda['order_id']) ?>">
                                                                <i class="ti ti-eye"></i> Ver Detalhes
                                                            </button>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <?php endforeach; ?>
                                        </div>

                                        <!-- Paginação -->
                                        <?php if ($totalPaginas > 1): ?>
                                        <nav aria-label="Paginação">
                                            <ul class="pagination justify-content-center">
                                                <?php if ($paginaAtual > 1): ?>
                                                <li class="page-item">
                                                    <a class="page-link"
                                                        href="?<?= http_build_query(array_merge($_GET, ['pagina' => $paginaAtual - 1])) ?>">
                                                        <i class="ti ti-chevron-left"></i> Anterior
                                                    </a>
                                                </li>
                                                <?php endif; ?>

                                                <?php for ($i = max(1, $paginaAtual - 2); $i <= min($totalPaginas, $paginaAtual + 2); $i++): ?>
                                                <li class="page-item <?= $i === $paginaAtual ? 'active' : '' ?>">
                                                    <a class="page-link"
                                                        href="?<?= http_build_query(array_merge($_GET, ['pagina' => $i])) ?>">
                                                        <?= $i ?>
                                                    </a>
                                                </li>
                                                <?php endfor; ?>

                                                <?php if ($paginaAtual < $totalPaginas): ?>
                                                <li class="page-item">
                                                    <a class="page-link"
                                                        href="?<?= http_build_query(array_merge($_GET, ['pagina' => $paginaAtual + 1])) ?>">
                                                        Próxima <i class="ti ti-chevron-right"></i>
                                                    </a>
                                                </li>
                                                <?php endif; ?>
                                            </ul>
                                        </nav>
                                        <?php endif; ?>

                                        <?php else: ?>
                                        <div class="text-center py-4">
                                            <i class="ti ti-file-x" style="font-size: 3rem; color: #ccc;"></i>
                                            <h5 class="mt-2">Nenhuma venda encontrada</h5>
                                            <p class="text-muted">Não há vendas que correspondam aos filtros aplicados
                                                para o fornecedor "<?= htmlspecialchars($fornecedorAtual) ?>".</p>
                                        </div>
                                        <?php endif; ?>

                                    </div>
                                </div>
                            </div>

                        </div>
                    </div>

                    <div class="content-backdrop fade"></div>
                </div>
                <!-- Content wrapper -->
            </div>
            <!-- / Layout page -->
        </div>

        <div class="layout-overlay layout-menu-toggle"></div>

        <div class="drag-target"></div>
    </div>
    <!-- / Layout wrapper -->

    <div class="modal fade" id="modalDetalhesVenda" tabindex="-1" aria-labelledby="modalDetalhesVendaLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="modalDetalhesVendaLabel">
                        <i class="ti ti-eye me-2"></i>Detalhes da Venda
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div id="modalDetalhesContent">
                        <div class="text-center py-4">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">Carregando...</span>
                            </div>
                            <p class="mt-2">Carregando detalhes da venda...</p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="ti ti-x me-1"></i>Fechar
                    </button>
                </div>
            </div>
        </div>
    </div>

    <div class="modal fade" id="modalReprocessar" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Reprocessar Etiquetas</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-0" style="height: 80vh;">
                    <iframe id="iframeReprocessar" src="" style="width: 100%; height: 100%; border: none;"></iframe>
                </div>
            </div>
        </div>
    </div>

   

    <script>
    $(document).ready(function() {
        // Botão Reprocessar Etiquetas
        const idsParaReprocessar = <?= json_encode($idsSemEtiqueta) ?>;
        $('#btnReprocessarEtiquetas').on('click', function() {
            const modal = new bootstrap.Modal(document.getElementById('modalReprocessar'));
            const iframe = document.getElementById('iframeReprocessar');
            
            // Set src to load the page
            iframe.src = '/dash/reprocessar_vendas';
            
            // When iframe loads, inject IDs
            iframe.onload = function() {
                try {
                    const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
                    // Tentar encontrar o textarea pelo ID ou tag
                    const textarea = iframeDoc.querySelector('textarea'); 
                    if (textarea) {
                        textarea.value = idsParaReprocessar.join('\n');
                    }
                } catch (e) {
                    console.error('Erro ao acessar iframe:', e);
                }
            };
            
            modal.show();
        });

        // Botão de gerar relatório
        $('#btnGerarRelatorio').on('click', function() {
            const btnText = $(this).html();
            const btn = $(this);
            
            // Coletar valores dos filtros
            const filtros = {
                data_de: $('#data_de').val(),
                data_ate: $('#data_ate').val(),
                integracao: $('#integracao').val(),
                status_pago: $('#status_pago').val(),
                id_venda: $('#id_venda').val()
            };

            // Verificar se há pelo menos um filtro
            const temFiltro = filtros.data_de || filtros.data_ate || filtros.integracao || 
                            filtros.status_pago || filtros.id_venda;

            if (!temFiltro) {
                Swal.fire({
                    icon: 'warning',
                    title: 'Atenção!',
                    text: 'É necessário selecionar pelo menos um filtro ou período para gerar o relatório!',
                    confirmButtonText: 'Entendi'
                });
                return;
            }

            // Desabilitar botão e mostrar loading
            btn.prop('disabled', true).html('<i class="ti ti-loader-2 ti-spin"></i> Gerando...');

            $.ajax({
                url: '/assets/ajax/vendas/gerar_relatorio_vendas.php',
                type: 'POST',
                data: filtros,
                dataType: 'json',
                success: function(response) {
                    btn.prop('disabled', false).html(btnText);
                    
                    if (response.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Relatório Gerado!',
                            html: `Relatório gerado com sucesso!<br><strong>${response.total_vendas}</strong> venda(s) exportada(s).`,
                            confirmButtonText: 'Baixar Relatório',
                            showCancelButton: true,
                            cancelButtonText: 'Fechar'
                        }).then((result) => {
                            if (result.isConfirmed) {
                                // Fazer download do arquivo
                                window.location.href = response.arquivo;
                            }
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Erro!',
                            text: response.message || 'Erro ao gerar relatório',
                            confirmButtonText: 'OK'
                        });
                    }
                },
                error: function(xhr, status, error) {
                    btn.prop('disabled', false).html(btnText);
                    
                    Swal.fire({
                        icon: 'error',
                        title: 'Erro!',
                        text: 'Erro ao gerar relatório. Tente novamente.',
                        confirmButtonText: 'OK'
                    });
                    console.error('Erro:', error);
                }
            });
        });

        // Atualizar totais em tempo real
        function atualizarResumo() {
            const totalVendas = <?= $totalGeralVendas ?>;
            const vendasPagas = <?= $totalVendasPagas ?>;
            const vendasNaoPagas = <?= $totalVendasNaoPagas ?>;
            const vendasSemEtiqueta = <?= $totalVendasSemEtiqueta ?>;

            // Adicionar resumo se não existir
            if ($('#resumoVendas').length === 0) {
                const resumoHtml = `
                        <div id="resumoVendas" class="row mb-3">
                            <div class="col-md-2">
                                <div class="card text-center">
                                    <div class="card-body p-2">
                                        <h5 class="card-title text-primary">${totalVendas}</h5>
                                        <p class="card-text small">Total de Vendas</p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="card text-center">
                                    <div class="card-body p-2">
                                        <h5 class="card-title text-success">${vendasPagas}</h5>
                                        <p class="card-text small">Vendas Pagas</p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="card text-center">
                                    <div class="card-body p-2">
                                        <h5 class="card-title text-warning">${vendasNaoPagas}</h5>
                                        <p class="card-text small">Vendas Não Pagas</p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="card text-center cursor-pointer" onclick="window.location.href='?sem_etiqueta=1'" style="cursor: pointer;">
                                    <div class="card-body p-2">
                                        <h5 class="card-title text-danger">${vendasSemEtiqueta}</h5>
                                        <p class="card-text small">Vendas Sem Etiquetas</p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="card text-center">
                                    <div class="card-body p-2">
                                        <h5 class="card-title text-info"><?= $totalPaginas ?></h5>
                                        <p class="card-text small">Páginas</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    `;
                $('hr').after(resumoHtml);
            }
        }

        // Chamar função ao carregar a página
        atualizarResumo();

        // Ver detalhes individual
        $('.ver-detalhes').on('click', function() {
            const saleTitle = $(this).data('sale-title');
            verDetalhesVenda(saleTitle);
        });

        // Carregar miniaturas de produtos para cada venda (lazy simples ao carregar a página)
        const urlBaseProdutos = "<?= addslashes(Config::get('site_configs.website_url')) ?>";
        $('.miniaturas-produtos').each(function() {
            const container = $(this);
            const saleTitle = container.data('sale-title');
            // Evitar requisições duplicadas
            if (!saleTitle) return;
            $.ajax({
                url: '/assets/ajax/vendas/buscar_produtos_venda.php',
                type: 'POST',
                dataType: 'json',
                data: {
                    saleTitle: saleTitle
                },
                success: function(res) {
                    const produtos = res.produtos || [];
                    if (produtos.length === 0) {
                        container.html(
                        '<span class="text-muted small">Sem produtos</span>');
                        return;
                    }
                    const maxThumbs = 5;
                    let html = '<div class="thumbs-wrapper">';
                    produtos.slice(0, maxThumbs).forEach(function(p, idx) {
                        const imgSrc = urlBaseProdutos +
                            '/images/produtos_imagens/' + (p.product_id || p.id ||
                                '0') + '/thumb.png';
                        const titulo = (p.product_title || 'Produto');
                        const tituloLimitado = titulo.length > 40 ? titulo
                            .substring(0, 40) + '…' : titulo;
                        const qtd = p.quantidade || p.qty || 0;
                        html += `
                                <div class="mini-thumb" data-bs-toggle="tooltip" data-bs-placement="top" title="${titulo.replace(/"/g,'&quot;')} (Qtd: ${qtd})">
                                    <img src="${imgSrc}" alt="${tituloLimitado}" onerror="this.style.display='none';this.parentNode.classList.add('sem-imagem');">
                                    <span class="qtd">${qtd}</span>
                                </div>`;
                    });
                    if (produtos.length > maxThumbs) {
                        const restante = produtos.length - maxThumbs;
                        html +=
                            `<div class="mini-thumb mais" data-bs-toggle="tooltip" data-bs-placement="top" title="+${restante} produto(s)">+${restante}</div>`;
                    }
                    html += '</div>';
                    container.html(html);
                    inicializarTooltips();
                },
                error: function() {
                    container.html(
                        '<span class="text-danger small">Erro ao carregar</span>');
                }
            });
        });

        function inicializarTooltips() {
            const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            tooltipTriggerList.map(function(tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
        }

        // Função para ver detalhes da venda
        function verDetalhesVenda(saleTitle) {
            // Abrir modal e mostrar loading
            $('#modalDetalhesVenda').modal('show');
            $('#modalDetalhesContent').html(`
                    <div class="text-center py-4">
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden">Carregando...</span>
                        </div>
                        <p class="mt-2">Carregando detalhes da venda...</p>
                    </div>
                `);

            // Buscar a venda nos dados PHP carregados (apenas para fallback)
            <?php
                echo "const vendasData = " . json_encode($vendas) . ";\n";
                ?>

            console.log('Buscando venda com saleTitle:', saleTitle);

            // Buscar dados da venda e produtos via AJAX
            $.ajax({
                url: '/assets/ajax/vendas/buscar_produtos_venda.php',
                type: 'POST',
                data: {
                    saleTitle: saleTitle
                },
                dataType: 'json',
                success: function(response) {
                    console.log('Resposta do AJAX:', response);
                    console.log('Debug info:', response.debug);

                    // Usar dados da venda retornados pelo AJAX, ou fallback para dados locais
                    const venda = response.venda || vendasData.find(v => v.saleTitle === saleTitle);

                    if (response.produtos) {
                        exibirDetalhesVenda(venda, response.produtos);
                    } else {
                        exibirDetalhesVenda(venda, []);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Erro no AJAX:', {
                        status: status,
                        error: error,
                        responseText: xhr.responseText,
                        statusCode: xhr.status
                    });

                    // Em caso de erro no AJAX, tentar usar dados locais como fallback
                    const venda = vendasData.find(v => v.saleTitle === saleTitle);
                    exibirDetalhesVenda(venda, []);
                }
            });

        }

        // Função para exibir os detalhes da venda no modal
        function exibirDetalhesVenda(venda, produtos) {
            // Verificar se venda existe
            if (!venda) {
                $('#modalDetalhesContent').html(`
                        <div class="text-center py-4">
                            <i class="ti ti-alert-circle text-warning" style="font-size: 3rem;"></i>
                            <h5 class="mt-2">Venda não encontrada</h5>
                            <p class="text-muted">Não foi possível carregar os detalhes da venda.</p>
                        </div>
                    `);
                return;
            }

            let produtosHtml = '';
            let totalProdutos = 0;

            if (produtos && produtos.length > 0) {
                produtos.forEach(function(produto) {
                    const subtotal = parseFloat(produto.vl_custo || 0) * parseInt(produto.quantidade || 0);
                    const url_base = "<?= addslashes(Config::get('site_configs.website_url')) ?>";
                    const thumb = url_base + '/images/produtos_imagens/' + produto.product_id +
                        '/thumb.png';

                    totalProdutos += subtotal;

                    produtosHtml += `
                            <tr>
                                <td>
                                    <div class="d-flex align-items-center">
                                        ${thumb ? 
                                            `<img src="${thumb}" alt="Produto" class="rounded me-2" style="width: 40px; height: 40px; object-fit: cover;">` : 
                                            '<div class="bg-light rounded me-2 d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;"><i class="ti ti-package text-muted"></i></div>'
                                        }
                                        <div>
                                            <div class="fw-medium">${produto.product_title && produto.product_title.length > 35 ? produto.product_title.substring(0, 35) + '...' : (produto.product_title || 'N/A')}</div>
                                            <small class="text-muted">SKU: ${produto.sku || 'N/A'}</small>
                                        </div>
                                    </div>
                                </td>
                                <td class="text-center">${produto.quantidade || 0}</td>
                                <td class="text-end">R$ ${parseFloat(produto.vl_custo || 0).toLocaleString('pt-BR', {minimumFractionDigits: 2})}</td>
                                <td class="text-end">
                                    ${produto.preco_custo_forn ? 
                                        `R$ ${parseFloat(produto.preco_custo_forn).toLocaleString('pt-BR', {minimumFractionDigits: 2})}` : 
                                        '<span class="text-muted">N/A</span>'
                                    }
                                </td>
                                <td class="text-end fw-medium">R$ ${subtotal.toLocaleString('pt-BR', {minimumFractionDigits: 2})}</td>
                            </tr>
                        `;
                });
            } else {
                produtosHtml = `
                        <tr>
                            <td colspan="5" class="text-center text-muted py-4">
                                <i class="ti ti-package-off mb-2" style="font-size: 2rem;"></i>
                                <br>Nenhum produto encontrado
                            </td>
                        </tr>
                    `;
            }

            const content = `
                    <div class="row">
                        <!-- Informações da Venda -->
                        <div class="col-12 mb-4">
                            <div class="card border shadow-sm">
                                <div class="card-header border-bottom">
                                    <h6 class="card-title mb-0 fw-bold">
                                        <i class="ti ti-shopping-cart me-2 text-primary"></i>Informações da Venda
                                    </h6>
                                </div>
                                <div class="card-body p-4">
                                    <div class="row g-4">
                                        <div class="col-md-6">
                                            <div class="info-box p-3 border rounded h-100">
                                                <label class="form-label fw-semibold text-muted small mb-2">ID da Venda:</label>
                                                <div class="fw-bold fs-6">${venda.saleTitle || 'N/A'}</div>
                                            </div>
                                        </div>

                                        <div class="col-md-6">
                                            <div class="info-box p-3 border rounded h-100">
                                                <label class="form-label fw-semibold text-muted small mb-2">Código de Rastreio:</label>
                                                <div class="fw-bold fs-6">${venda.codigo_rastreio || 'N/A'}</div>
                                            </div>
                                        </div>

                                        <div class="col-md-6">
                                            <div class="info-box p-3 border rounded h-100">
                                                <label class="form-label fw-semibold text-muted small mb-2">Integração:</label>
                                                <div><span class="badge bg-primary fs-6">${venda.integracao || 'N/A'}</span></div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="info-box p-3 border rounded h-100">
                                                <label class="form-label fw-semibold text-muted small mb-2">Tipo de Envio:</label>
                                                <div><span class="badge bg-info fs-6">${venda.tipo_envio || 'N/A'}</span></div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="info-box p-3 border rounded h-100">
                                                <label class="form-label fw-semibold text-muted small mb-2">Valor Total:</label>
                                                <div class="text-success fw-bold fs-5">
                                                    R$ ${parseFloat(venda.vlr_total || 0).toLocaleString('pt-BR', {minimumFractionDigits: 2})}
                                                </div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="info-box p-3 border rounded h-100">
                                                <label class="form-label fw-semibold text-muted small mb-2">Data do Pagamento:</label>
                                                <div class="fw-medium">
                                                    ${venda.dta_vl_pago ? new Date(venda.dta_vl_pago).toLocaleString('pt-BR') : '<span class="text-muted">N/A</span>'}
                                                </div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="info-box p-3 border rounded h-100">
                                                <label class="form-label fw-semibold text-muted small mb-2">Data do Envio:</label>
                                                <div class="fw-medium">
                                                    ${venda.dta_enviado ? new Date(venda.dta_enviado).toLocaleString('pt-BR') : '<span class="text-muted">N/A</span>'}
                                                </div>
                                            </div>
                                        </div>
                                        ${venda.etiqueta_ml ? `
                                        <div class="col-12">
                                            <div class="info-box p-3 border rounded">
                                                <label class="form-label fw-semibold text-muted small mb-2">URL da Etiqueta:</label>
                                                <div class="mt-2">
                                                    ${(venda.observacoes && venda.observacoes.includes('ERRO ETIQUETA MercadoLivre')) ? `
                                                        <span class="text-danger fw-bold">
                                                            Sem etiqueta, envie manualmente e leia o motivo nas observações.
                                                        </span>
                                                    ` : (venda.etiqueta_ml === 'buffering' || venda.etiqueta_ml === 'buffered') ? `
                                                        <span class="badge bg-warning">
                                                            <i class="ti ti-clock me-1"></i>Aguardando liberação do marketplace
                                                        </span>
                                                    ` : (venda.etiqueta_ml === 'processing_label') ? `
                                                        <span class="badge bg-warning">
                                                            <i class="ti ti-alert-circle me-1"></i>Sem etiqueta
                                                        </span>
                                                    ` : `
                                                        <a href="${venda.etiqueta_ml}" target="_blank" class="btn btn-outline-primary btn-sm">
                                                            <i class="ti ti-external-link me-1"></i>
                                                            Visualizar Etiqueta
                                                        </a>
                                                    `}
                                                </div>
                                                ${venda.observacoes ? `
                                                <div class="mt-3 pt-3 border-top">
                                                    <label class="form-label fw-semibold text-muted small mb-2">Observação:</label>
                                                    <div class="text-danger fw-bold">
                                                        ${venda.observacoes}
                                                    </div>
                                                </div>
                                                ` : ''}
                                            </div>
                                        </div>
                                        ` : ''}
                                        ${venda.observacoes ? `
                                        <div class="col-12">
                                            <div class="info-box p-3 border rounded">
                                                <label class="form-label fw-semibold text-muted small mb-2">Observação:</label>
                                                <div class="mt-2 text-danger fw-bold">
                                                    ${venda.observacoes}
                                                </div>
                                            </div>
                                        </div>
                                        ` : ''}
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Status da Venda -->
                        <div class="col-12 mb-4">
                            <div class="card border shadow-sm">
                                <div class="card-header border-bottom d-flex justify-content-between align-items-center">
                                    <h6 class="card-title mb-0 fw-bold">
                                        <i class="ti ti-flag me-2 text-info"></i>Status da Venda
                                    </h6>
                                    ${venda.vlr_custo_pago != 1 ? `
                                        <button class="btn btn-sm btn-success" onclick="marcarComoPago('${venda.saleTitle}')">
                                            <i class="ti ti-check me-1"></i>Marcar como Pago
                                        </button>
                                    ` : ''}
                                </div>
                                <div class="card-body p-4">
                                    <div class="row g-4">
                                        <div class="col-md-4">
                                            <div class="status-box p-4 border rounded text-center h-100">
                                                <div class="status-icon mb-3">
                                                    ${venda.vlr_custo_pago == 1 ? 
                                                        '<i class="ti ti-circle-check text-success" style="font-size: 3.5rem;"></i>' : 
                                                        '<i class="ti ti-circle-x text-danger" style="font-size: 3.5rem;"></i>'
                                                    }
                                                </div>
                                                <h6 class="mb-3 fw-bold fs-5">Custo Pago</h6>
                                                <span class="badge ${venda.vlr_custo_pago == 1 ? 'bg-success' : 'bg-danger'} fs-6 px-3 py-2 mb-3">
                                                    ${venda.vlr_custo_pago == 1 ? 'Pago' : 'Não Pago'}
                                                </span>
                                                ${venda.vlr_custo_pago == 1 && venda.dta_vl_pago ? `
                                                    <div class="text-muted small mt-2 border-top pt-2">
                                                        ${new Date(venda.dta_vl_pago).toLocaleString('pt-BR')}
                                                    </div>
                                                ` : ''}
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="status-box p-4 border rounded text-center h-100">
                                                <div class="status-icon mb-3">
                                                    ${venda.enviado == 1 ? 
                                                        '<i class="ti ti-truck text-info" style="font-size: 3.5rem;"></i>' : 
                                                        '<i class="ti ti-clock text-warning" style="font-size: 3.5rem;"></i>'
                                                    }
                                                </div>
                                                <h6 class="mb-3 fw-bold fs-5">Envio</h6>
                                                <span class="badge ${venda.enviado == 1 ? 'bg-info' : 'bg-warning'} fs-6 px-3 py-2 mb-3">
                                                    ${venda.enviado == 1 ? 'Enviado' : 'Pendente'}
                                                </span>
                                                ${venda.enviado == 1 && venda.dta_enviado ? `
                                                    <div class="text-muted small mt-2 border-top pt-2">
                                                        ${new Date(venda.dta_enviado).toLocaleString('pt-BR')}
                                                    </div>
                                                ` : ''}
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="status-box p-4 border rounded text-center h-100">
                                                <div class="status-icon mb-3">
                                                    ${(venda.reembolsado == 1 || venda.refunded == 1) ? 
                                                        '<i class="ti ti-receipt-refund text-danger" style="font-size: 3.5rem;"></i>' : 
                                                        '<i class="ti ti-receipt-refund text-secondary" style="font-size: 3.5rem;"></i>'
                                                    }
                                                </div>
                                                <h6 class="mb-3 fw-bold fs-5">Reembolso</h6>
                                                <span class="badge ${(venda.reembolsado == 1 || venda.refunded == 1) ? 'bg-danger' : 'bg-secondary'} fs-6 px-3 py-2 mb-3">
                                                    ${(venda.reembolsado == 1 || venda.refunded == 1) ? 'Reembolsado' : '--'}
                                                </span>
                                                ${(venda.reembolsado == 1 || venda.refunded == 1) && (venda.dta_reembolso || venda.refund_date) ? `
                                                    <div class="text-muted small mt-2 border-top pt-2">
                                                        ${new Date(venda.dta_reembolso || venda.refund_date).toLocaleString('pt-BR')}
                                                    </div>
                                                ` : ''}
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-12 mb-4">
                            <div class="card border shadow-sm">
                                <div class="card-header border-bottom">
                                    <h6 class="card-title mb-0 fw-bold">
                                        <i class="ti ti-user me-2 text-success"></i>Informações do Vendedor
                                    </h6>
                                </div>
                                <div class="card-body p-4">
                                    <div class="row g-4">
                                        <div class="col-md-6">
                                            <div class="info-box p-3 border rounded h-100">
                                                <div class="d-flex align-items-center">
                                                    <div class="flex-shrink-0 me-3">
                                                        <div class="icon-circle bg-success text-white rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                                            <i class="ti ti-user"></i>
                                                        </div>
                                                    </div>
                                                    <div class="flex-grow-1">
                                                        <label class="form-label fw-semibold text-muted small mb-1">Nome:</label>
                                                        <div class="fw-bold">${venda.vendedor_nome || 'N/A'}</div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="info-box p-3 border rounded h-100">
                                                <div class="d-flex align-items-center">
                                                    <div class="flex-shrink-0 me-3">
                                                        <div class="icon-circle bg-primary text-white rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                                            <i class="ti ti-mail"></i>
                                                        </div>
                                                    </div>
                                                    <div class="flex-grow-1">
                                                        <label class="form-label fw-semibold text-muted small mb-1">E-mail:</label>
                                                        <div class="fw-medium">${venda.vendedor_email || 'N/A'}</div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="info-box p-3 border rounded h-100">
                                                <div class="d-flex align-items-center">
                                                    <div class="flex-shrink-0 me-3">
                                                        <div class="icon-circle bg-info text-white rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                                            <i class="ti ti-phone"></i>
                                                        </div>
                                                    </div>
                                                    <div class="flex-grow-1">
                                                        <label class="form-label fw-semibold text-muted small mb-1">Telefone:</label>
                                                        <div class="fw-medium">${venda.vendedor_celular || 'N/A'}</div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="info-box p-3 border rounded h-100">
                                                <div class="d-flex align-items-center">
                                                    <div class="flex-shrink-0 me-3">
                                                        <div class="icon-circle bg-warning text-white rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                                            <i class="ti ti-building-store"></i>
                                                        </div>
                                                    </div>
                                                    <div class="flex-grow-1">
                                                        <label class="form-label fw-semibold text-muted small mb-1">Loja:</label>
                                                        <div class="fw-medium">${venda.loja_nome || 'N/A'}</div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Pagamento -->
                        <div class="col-12 mb-4">
                            <div class="card border shadow-sm">
                                <div class="card-header border-bottom">
                                    <h6 class="card-title mb-0 fw-bold">
                                        <i class="ti ti-cash me-2 text-warning"></i>Pagamento da Venda <small class="text-muted"> - (Preço de custo)</small>
                                    </h6>
                                </div>
                                <div class="card-body p-4" id="payment-section-${venda.saleTitle}">
                                    <!-- Detalhes do método de pagamento (carregado via AJAX) -->
                                    <div id="payment-details-${venda.saleTitle}" class="mt-4">
                                        <div class="text-center py-3">
                                            <div class="spinner-border spinner-border-sm text-primary" role="status">
                                                <span class="visually-hidden">Carregando método de pagamento...</span>
                                            </div>
                                            <p class="mt-2 small text-muted">Analisando método de pagamento...</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Produtos da Venda -->
                        <div class="col-12">
                            <div class="card border shadow-sm">
                                <div class="card-header border-bottom d-flex justify-content-between align-items-center">
                                    <h6 class="card-title mb-0 fw-bold">
                                        <i class="ti ti-package me-2 text-secondary"></i>Produtos da Venda <small class="text-muted">- (Dados de venda)</small>
                                    </h6>
                                    <span class="badge bg-secondary fs-6 px-3 py-2">${produtos ? produtos.length : 0} item(s)</span>
                                </div>
                                <div class="card-body p-0">
                                    <div class="table-responsive">
                                        <table class="table table-hover mb-0">
                                            <thead class="table-dark">
                                                <tr>
                                                    <th class="fw-bold border-0">Produto</th>
                                                    <th class="text-center fw-bold border-0">Qtd</th>
                                                    <th class="text-end fw-bold border-0">Valor Unit.</th>
                                                    <th class="text-end fw-bold border-0">Custo Fornecedor</th>
                                                    <th class="text-end fw-bold border-0">Subtotal</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                ${produtosHtml}
                                            </tbody>
                                            ${produtos && produtos.length > 0 ? `
                                            <tfoot class="table-light border-top">
                                                <tr>
                                                    <th colspan="4" class="text-end fw-bold fs-5 py-3">Total dos Produtos:</th>
                                                    <th class="text-end text-success fw-bold fs-4 py-3">
                                                        R$ ${totalProdutos.toLocaleString('pt-BR', {minimumFractionDigits: 2})}
                                                    </th>
                                                </tr>
                                            </tfoot>
                                            ` : ''}
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                `;

            $('#modalDetalhesContent').html(content);
            
            // Carregar detalhes do método de pagamento
            carregarDetalhesMetodoPagamento(venda.saleTitle);
        }

        // Função para carregar detalhes do método de pagamento
        function carregarDetalhesMetodoPagamento(saleTitle) {
            $.ajax({
                url: '/assets/ajax/vendas/buscar_metodo_pagamento.php',
                type: 'POST',
                data: {
                    id_venda: saleTitle
                },
                dataType: 'json',
                success: function(response) {
                    console.log('Resposta completa da API:', response);
                    console.log('Debug da API:', response.debug);
                    console.log('Método de pagamento detectado:', response.metodo_pagamento);
                    
                    let paymentHtml = '';
                    
                    if (response.success) {
                        const metodoPagamento = response.metodo_pagamento;
                        console.log('Tipo do método:', metodoPagamento.tipo);
                        console.log('Tem transação carteira:', response.debug?.tem_transacao_carteira);
                        console.log('Tem registro pedidos:', response.debug?.tem_registro_pedidos);
                        
                        switch(metodoPagamento.tipo) {
                            case 'carteira_completa':
                                paymentHtml = `
                                    <div class="payment-method-details border rounded p-4">
                                        <div class="row align-items-center">
                                            <div class="col-md-2 text-center">
                                                <div class="payment-method-icon">
                                                    <i class="ti ti-wallet text-primary" style="font-size: 2.5rem;"></i>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <h6 class="mb-2 fw-bold text-primary">
                                                    <i class="ti ti-wallet me-2"></i>Pagamento via Carteira
                                                </h6>
                                                <p class="text-muted mb-0 small">
                                                    Pagamento realizado integralmente usando saldo da carteira digital
                                                </p>
                                                ${metodoPagamento.carteira_id ? `
                                                <div class="mt-2">
                                                    <small class="text-muted">
                                                        <strong>ID Pagamento:</strong> <a href="/dash/pagamentos?i=${btoa(metodoPagamento.carteira_id)}" target="_blank">${metodoPagamento.carteira_id}</a> <br>
                                                        <strong>ID Carteira:</strong> ${metodoPagamento.carteira_id_db}
                                                    </small>
                                                </div>
                                                ` : ''}
                                            </div>
                                            <div class="col-md-4 text-end">
                                                <div class="payment-amount">
                                                    <span class="fw-bold text-primary fs-5">
                                                        R$ ${parseFloat(metodoPagamento.valor_total || 0).toLocaleString('pt-BR', {minimumFractionDigits: 2})}
                                                    </span>
                                                </div>
                                                <small class="text-muted">Valor debitado da carteira</small>
                                            </div>
                                        </div>
                                    </div>
                                `;
                                break;
                                
                            case 'pix_shipay':
                                paymentHtml = `
                                    <div class="payment-method-details border rounded p-4">
                                        <div class="row align-items-center">
                                            <div class="col-md-2 text-center">
                                                <div class="payment-method-icon">
                                                    <i class="ti ti-qrcode text-success" style="font-size: 2.5rem;"></i>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <h6 class="mb-2 fw-bold text-success">
                                                    <i class="ti ti-qrcode me-2"></i>Pagamento via PIX 
                                                </h6>
                                                <p class="text-muted mb-0 small">
                                                    Pagamento realizado via PIX através da plataforma 
                                                </p>
                                                ${metodoPagamento.shipay_id ? `
                                                <div class="mt-2">
                                                    <small class="text-muted">
                                                        <strong>ID Shipay:</strong> ${metodoPagamento.shipay_id}
                                                    </small>
                                                </div>
                                                ` : ''}
                                            </div>
                                            <div class="col-md-4 text-end">
                                                <div class="payment-amount">
                                                    <span class="fw-bold text-success fs-5">
                                                        R$ ${parseFloat(metodoPagamento.valor_total || 0).toLocaleString('pt-BR', {minimumFractionDigits: 2})}
                                                    </span>
                                                </div>
                                                <small class="text-muted">Valor via PIX</small>
                                            </div>
                                        </div>
                                    </div>
                                `;
                                break;
                                
                            case 'misto':
                                paymentHtml = `
                                    <div class="payment-method-details border rounded p-4">
                                        <div class="row">
                                            <div class="col-12 mb-3">
                                                <h6 class="mb-0 fw-bold text-info">
                                                    <i class="ti ti-credit-card me-2"></i>Pagamento Misto (Carteira + PIX)
                                                </h6>
                                                <p class="text-muted mb-0 small">
                                                    <strong>R$ ${parseFloat(metodoPagamento.valor_carteira || 0).toLocaleString('pt-BR', {minimumFractionDigits: 2})}</strong> 
                                                    foi debitado da carteira e 
                                                    <strong>R$ ${parseFloat(metodoPagamento.valor_pix || 0).toLocaleString('pt-BR', {minimumFractionDigits: 2})}</strong> 
                                                    foi pago via PIX
                                                </p>
                                            </div>
                                        </div>
                                        <div class="row g-3">
                                            <div class="col-md-6">
                                                <div class="payment-part border rounded p-3">
                                                    <div class="d-flex align-items-center mb-2">
                                                        <i class="ti ti-wallet text-primary me-2" style="font-size: 1.5rem;"></i>
                                                        <strong class="text-primary">Carteira Digital</strong>
                                                    </div>
                                                    <div class="payment-amount">
                                                        <span class="fw-bold text-primary fs-5">
                                                            R$ ${parseFloat(metodoPagamento.valor_carteira || 0).toLocaleString('pt-BR', {minimumFractionDigits: 2})}
                                                        </span>
                                                    </div>
                                                    <small class="text-muted">Valor debitado da carteira</small>
                                                    ${metodoPagamento.carteira_id ? `
                                                    <div class="mt-1">
                                                        <small class="text-muted d-block">
                                                            <strong>ID Pagamento:</strong> ${metodoPagamento.carteira_id}
                                                        </small>
                                                    </div>
                                                    ` : ''}
                                                    ${metodoPagamento.carteira_id_db ? `
                                                    <div class="mt-1">
                                                        <small class="text-muted d-block">
                                                            <strong>ID Carteira:</strong> ${metodoPagamento.carteira_id_db}
                                                        </small>
                                                    </div>
                                                    ` : ''}
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="payment-part border rounded p-3">
                                                    <div class="d-flex align-items-center mb-2">
                                                        <i class="ti ti-qrcode text-success me-2" style="font-size: 1.5rem;"></i>
                                                        <strong class="text-success">PIX </strong>
                                                    </div>
                                                    <div class="payment-amount">
                                                        <span class="fw-bold text-success fs-5">
                                                            R$ ${parseFloat(metodoPagamento.valor_pix || 0).toLocaleString('pt-BR', {minimumFractionDigits: 2})}
                                                        </span>
                                                    </div>
                                                    <small class="text-muted">Valor via PIX</small>
                                                    ${metodoPagamento.shipay_id ? `
                                                    <div class="mt-1">
                                                        <small class="text-muted d-block">
                                                            <strong>ID:</strong> ${metodoPagamento.shipay_id}
                                                        </small>
                                                    </div>
                                                    ` : ''}
                                                </div>
                                            </div>
                                        </div>
                                        <div class="row mt-3">
                                            <div class="col-12">
                                                <div class="total-summary text-center p-2 bg-light rounded">
                                                    <strong class="text-info">
                                                        Total: R$ ${parseFloat(metodoPagamento.valor_total || 0).toLocaleString('pt-BR', {minimumFractionDigits: 2})}
                                                    </strong>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                `;
                                break;
                                
                            default:
                                paymentHtml = `
                                    <div class="payment-method-details border rounded p-4 text-center">
                                        <i class="ti ti-help-circle text-muted" style="font-size: 2rem;"></i>
                                        <h6 class="mt-2 text-muted">Método de pagamento não identificado</h6>
                                        <p class="small text-muted mb-0">Não foi possível determinar o método de pagamento usado</p>
                                    </div>
                                `;
                        }
                    } else {
                        paymentHtml = `
                            <div class="payment-method-details border rounded p-4 text-center">
                                <i class="ti ti-alert-triangle text-warning" style="font-size: 2rem;"></i>
                                <h6 class="mt-2 text-warning">Erro ao carregar método de pagamento</h6>
                                <p class="small text-muted mb-0">${response.message || 'Erro desconhecido'}</p>
                            </div>
                        `;
                    }
                    
                    $(`#payment-details-${saleTitle}`).html(paymentHtml);
                },
                error: function(xhr, status, error) {
                    console.error('Erro ao buscar método de pagamento:', error);
                    $(`#payment-details-${saleTitle}`).html(`
                        <div class="payment-method-details border rounded p-4 text-center">
                            <i class="ti ti-wifi-off text-danger" style="font-size: 2rem;"></i>
                            <h6 class="mt-2 text-danger">Erro de conexão</h6>
                            <p class="small text-muted mb-0">Não foi possível carregar os detalhes do pagamento</p>
                        </div>
                    `);
                }
            });
        }

        function limitarTexto(texto, limite) {
            return texto.length > limite ? texto.substring(0, limite) + '...' : texto;
        }

        // Adicionar tooltip para títulos longos
        $('.card-title').each(function() {
            if ($(this).text().includes('...')) {
                $(this).attr('title', $(this).data('full-title') || $(this).text());
            }
        });
    });

    // Função para marcar venda como paga manualmente
    function marcarComoPago(saleTitle) {
        Swal.fire({
            title: 'Você tem certeza?',
            text: 'Você tem certeza que deseja marcar como pago?',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#28c76f',
            cancelButtonColor: '#d33',
            confirmButtonText: 'Sim, marcar como pago!',
            cancelButtonText: 'Cancelar',
            customClass: {
                container: 'swal-modal-overlay'
            }
        }).then((result) => {
            if (result.isConfirmed) {
                // Mostrar loading
                Swal.fire({
                    title: 'Processando...',
                    text: 'Marcando venda como paga...',
                    allowOutsideClick: false,
                    allowEscapeKey: false,
                    customClass: {
                        container: 'swal-modal-overlay'
                    },
                    didOpen: () => {
                        Swal.showLoading();
                    }
                });

                $.ajax({
                    url: '/assets/ajax/vendas/marcar_como_pago.php',
                    type: 'POST',
                    data: {
                        saleTitle: saleTitle
                    },
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Sucesso!',
                                text: response.message,
                                confirmButtonText: 'OK',
                                customClass: {
                                    container: 'swal-modal-overlay'
                                }
                            }).then(() => {
                                // Recarregar modal com dados atualizados
                                $('#modalDetalhesVenda').modal('hide');
                                // Recarregar página para atualizar tabela
                                window.location.reload();
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Erro!',
                                text: response.message,
                                confirmButtonText: 'OK',
                                customClass: {
                                    container: 'swal-modal-overlay'
                                }
                            });
                        }
                    },
                    error: function(xhr, status, error) {
                        Swal.fire({
                            icon: 'error',
                            title: 'Erro!',
                            text: 'Erro ao marcar venda como paga. Tente novamente.',
                            confirmButtonText: 'OK',
                            customClass: {
                                container: 'swal-modal-overlay'
                            }
                        });
                        console.error('Erro:', error);
                    }
                });
            }
        });
    }

    function logout() {
        Swal.fire({
            toast: true,
            position: "top-end",
            icon: 'success',
            title: 'Você se desconectou com sucesso! Até logo. 😉',
            showConfirmButton: false,
            timer: 2000,
        }).then(function() {
            window.location.href = '/logout';
        });
    }
    </script>

    <script src="../../assets/vendor/libs/jquery/jquery.js"></script>
    <script src="../../assets/vendor/libs/popper/popper.js"></script>
    <script src="../../assets/vendor/js/bootstrap.js"></script>
    <script src="../../assets/vendor/libs/node-waves/node-waves.js"></script>
    <script src="../../assets/vendor/libs/perfect-scrollbar/perfect-scrollbar.js"></script>
    <script src="../../assets/vendor/libs/hammer/hammer.js"></script>
    <script src="../../assets/vendor/libs/i18n/i18n.js"></script>
    <script src="../../assets/vendor/libs/typeahead-js/typeahead.js"></script>
    <script src="../../assets/vendor/js/menu.js"></script>
    <script src="../../assets/vendor/libs/apex-charts/apexcharts.js"></script>
    <script src="../../assets/vendor/libs/swiper/swiper.js"></script>
    <script src="../../assets/vendor/libs/flatpickr/flatpickr.js"></script>
    <script src="../../assets/js/main.js"></script>
    <script src="../../assets/js/dashboards-analytics.js"></script>
    <script src="../../assets/js/main.js"></script>
    <script src="../../assets/js/ui-modals.js"></script>
    <script src="../../assets/js/depositar_saldo.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script>
    (function() {
        if (typeof flatpickr !== 'undefined') {
            const start = '<?= htmlspecialchars($filtroDataDe) ?>';
            const end = '<?= htmlspecialchars($filtroDataAte) ?>';
            const defaults = [];
            if (start) {
                defaults.push(start);
            }
            if (end && end !== start) {
                defaults.push(end);
            }
            flatpickr('#periodo', {
                mode: 'range',
                dateFormat: 'Y-m-d',
                defaultDate: defaults,
                locale: {
                    firstDayOfWeek: 1
                },
                onReady: function(selDates, dateStr, instance) {
                    if (start && end) {
                        instance.input.value = start + ' até ' + end;
                    } else if (start) {
                        instance.input.value = start;
                    }
                },
                onChange: function(selectedDates) {
                    const pad = n => String(n).padStart(2, '0');
                    const fmt = d => d.getFullYear() + '-' + pad(d.getMonth() + 1) + '-' + pad(d
                        .getDate());
                    if (selectedDates.length === 2) {
                        document.getElementById('data_de').value = fmt(selectedDates[0]);
                        document.getElementById('data_ate').value = fmt(selectedDates[1]);
                        document.getElementById('periodo').value = fmt(selectedDates[0]) + ' até ' +
                            fmt(selectedDates[1]);
                    } else if (selectedDates.length === 1) {
                        document.getElementById('data_de').value = fmt(selectedDates[0]);
                        document.getElementById('data_ate').value = fmt(selectedDates[0]);
                    } else if (selectedDates.length === 0) {
                        document.getElementById('data_de').value = '';
                        document.getElementById('data_ate').value = '';
                    }
                }
            });
        }
    })();
    </script>

    <script>
    $(document).ready(function() {
        $('#vendas').DataTable({
            "language": {
                "url": "https://cdn.datatables.net/plug-ins/1.10.25/i18n/Portuguese.json"
            },
            "paging": false,
            "responsive": true,
            "searching": false
        });
    });

    function limitarTexto(texto, limite) {
        return texto.length > limite ? texto.substring(0, limite) + '...' : texto;
    }
    </script>

    <style>
    .miniaturas-produtos .thumbs-wrapper {
        display: flex;
        flex-wrap: wrap;
        gap: 4px;
    }

    .miniaturas-produtos .mini-thumb {
        position: relative;
        width: 34px;
        height: 34px;
        border-radius: 6px;
        overflow: hidden;
        background: #f3f4f6;
        border: 1px solid #e2e6eb;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 12px;
        font-weight: 600;
        color: #555;
    }

    .miniaturas-produtos .mini-thumb img {
        width: 100%;
        height: 100%;
        object-fit: cover;
        display: block;
    }

    .miniaturas-produtos .mini-thumb.sem-imagem {
        background: linear-gradient(135deg, #e8eaed, #dcdfe3);
        font-size: 10px;
        color: #777;
    }

    .miniaturas-produtos .mini-thumb .qtd {
        position: absolute;
        bottom: 0;
        right: 0;
        background: rgba(0, 0, 0, .55);
        color: #fff;
        font-size: 10px;
        line-height: 1;
        padding: 1px 3px;
        border-top-left-radius: 4px;
    }

    .miniaturas-produtos .mini-thumb.mais {
        background: #eef2f7;
        color: #333;
    }

    .miniaturas-produtos .mini-thumb.mais:hover {
        background: #e2e8f0;
    }

    .miniaturas-loading {
        font-size: 11px;
    }

    /* Estilos para organização dos boxes */
    .info-box, .status-box, .payment-box {
        min-height: 100px;
        display: flex;
        flex-direction: column;
        justify-content: center;
        transition: all 0.2s ease;
    }

    .info-box:hover, .status-box:hover, .payment-box:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.1);
    }

    .status-box {
        min-height: 200px;
    }

    .payment-box {
        min-height: 120px;
    }

    .icon-circle {
        min-width: 40px;
        min-height: 40px;
    }

    /* Garantir altura uniforme nos cards de status */
    .row .col-md-4 .status-box {
        height: 100%;
        min-height: 220px;
    }

    /* Garantir altura uniforme nos info-boxes */
    .row .col-md-6 .info-box {
        height: 100%;
        min-height: 90px;
    }

    /* Melhorar espaçamento das bordas */
    .card.border {
        border: 1px solid #dee2e6 !important;
        border-radius: 0.5rem;
    }

    .border.rounded {
        border: 1px solid #dee2e6 !important;
        border-radius: 0.375rem !important;
    }

    /* Estilos para métodos de pagamento */
    .payment-method-details {
        background: #f8f9fa;
        transition: all 0.2s ease;
    }

    .payment-method-details:hover {
        transform: translateY(-1px);
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    }

    .payment-method-icon {
        padding: 10px;
        background: rgba(255,255,255,0.8);
        border-radius: 12px;
        display: inline-block;
    }

    .payment-part {
        background: #ffffff;
        transition: all 0.2s ease;
    }

    .payment-part:hover {
        transform: translateY(-1px);
        box-shadow: 0 2px 6px rgba(0,0,0,0.1);
    }

    .total-summary {
        border: 2px dashed #17a2b8;
        background: #e8f4f8 !important;
    }

    /* Animação de loading para métodos de pagamento */
    .payment-loading {
        animation: pulse 1.5s infinite;
    }

    @keyframes pulse {
        0% { opacity: 1; }
        50% { opacity: 0.6; }
        100% { opacity: 1; }
    }

    /* SweetAlert acima do modal Bootstrap */
    .swal-modal-overlay {
        z-index: 10000 !important;
    }

    .swal2-container {
        z-index: 10000 !important;
    }
    </style>

</body>

</html>